/*
 SOFEXTimer Software Copyright (C) Hans-Joachim Oelkers 2017.
 
 Timer starts when liftoff occurs. This is indicated by a contact that is opened at lauch.
 When time elapsed, an igniter is burnt.
 Can be used e.g.
 - instead of deploying the parachute by motor exhaust
 - for an airstart

 HINTS
 A contact has to be provided that is closed at first and opens at lauch. Typically a paperclip is
 used for this purpose.
 
 ATTENTION: If the contact is not closed at switch-on, nothing will happen, so a rocket may fall down later.
 This is to avoid burning the igniter soon after switch-on, when a person doing some handling of the rocket
 maybe stands close to the lauchner!
 
 To the Pyro contacts only a so-called 'bridge igniter A' may be connected.
 There is no protection against connecting a battery the wrong way, i.e. interchanging + and - pole!
 The battery voltage may have 3.7 to 5.5 V, Therefore, a 3.7 V (1S) LiPo battery is strongly recommended.
 There is no provision against damage by connecting a battery with a voltage higher than 5.5 V!
 
 DISCLAIMER
 This program is part of an introduction course into ATtiny programming. The SOFEX-Timer therefore is
 course material, not a commercial product! Using it is at the user's own risk. Especially it must not
 be used with Mid or High Power Rockets, as it lacks important features like 
 - Arming Switch provision 
 - check of igniter continuity
 - provision for connecting a battery the wrong way, i.e. interchanging + and - pole
 - provision against connecting a battery with a too high voltage etc.
 Although the program was carefully written and tested, I don't take any responsibility for
 possible damages that may occur by using the software and/or hardware. 
 Be warned that batteries, especially LiPo batteries may get hot and possibly burn in such a case.
 Also, rockets may cause damage to people and any belongings in case they fall down in an unexpected way.
*/



#include <SoftwareSerial.h>

// Definitions for Serial debug
#define rxPin 7
#define txPin 3

SoftwareSerial mySerial(rxPin, txPin);    // for test print

boolean contactWasClosed = false; // status of the contact to open at launch
boolean startDetected = false;    // start has not yet been detected
boolean timeElapsed = false;      // time has not yet been elapsed

long startTime;
long pyroTimeSec;                 // time read from jumper settings in secs
long pyroTime;                    // time in ms


/*          ATtiny84
 *        +----  ----+
 *    VCC |          | GND  
 * Pin 10 |          | Pin 0      
 *  Pin 9 |          | Pin 1
 *  Reset |          | Pin 2
 *  Pin 8 |          | Pin 3
 *  Pin 7 |          | Pin 4
 *  Pin 6 |          | Pin 5
 *        +----------+
 */
 
// define pins used
const int pinTime3 = 2;
const int pinTime2 = 1;
const int pinTime1 = 9;
const int pinTime0 = 10;
const int pinPyro = 0;
const int pinContact = 4;

// digits of the jumper settings
int digit3;
int digit2;
int digit1;
int digit0;


void setup()
{
  mySerial.begin(9600);

  // just to clear a possibly connected 2-line display
  mySerial.println();
  mySerial.println();
  
  //Initialise input pins
  pinMode(pinTime3, INPUT_PULLUP);
  pinMode(pinTime2, INPUT_PULLUP);
  pinMode(pinTime1, INPUT_PULLUP);
  pinMode(pinTime0, INPUT_PULLUP);
  pinMode(pinContact, INPUT_PULLUP);
  pinMode(rxPin, INPUT);              // we don't use the RX pin, just to show it
  
  // Initialise output pins
  pinMode(pinPyro, OUTPUT);
  pinMode(txPin, OUTPUT);
  
 
  //read time value from jumper settings
  if (digitalRead(pinTime3) == 0)
    digit3 = 1;
  else
    digit3 = 0;
  if (digitalRead(pinTime2) == 0)
    digit2 = 1;
  else
    digit2 = 0;
  if (digitalRead(pinTime1) == 0)
    digit1 = 1;
  else
    digit1 = 0;
    if (digitalRead(pinTime0) == 0)
    digit0 = 1;
  else
    digit0 = 0;

  // input of the jumper switches is finished
  // we "remove" the internal pullups to save power, although this is minimal
  pinMode(pinTime3, INPUT);
  pinMode(pinTime2, INPUT);
  pinMode(pinTime1, INPUT);
  pinMode(pinTime0, INPUT);
  
  pyroTimeSec = (8 * digit3 + 4 * digit2 + 2 * digit1 + digit0);  // time in s
  pyroTime = pyroTimeSec * 1000;                                  // time in ms
  mySerial.print("Time(s) = ");    // for test
  mySerial.println(pyroTimeSec);       // for test
  
  if (digitalRead(pinContact) == 1)
  {
    // contact was open at start! we do nothing to avoid early pyro ignition!
    contactWasClosed = false;
  }
  else
    contactWasClosed = true;
}

void loop()
{
  if (contactWasClosed)             // avoid ignition if contact wasn't closed at start
  {
    if (!startDetected)
    {
      if (digitalRead(pinContact) == 1)
      {
        startDetected = true;      //contact opened: start has been detected
        startTime = millis();      //now time is running
        mySerial.println("Contact opened!");   // for test
      }
    }
    if (startDetected && (!timeElapsed) && ((millis() - startTime) > pyroTime))
    {
      // time has elapsed now, we want to do the ignition just once
      mySerial.println("Ignition!");    // for test
      digitalWrite(pinPyro, HIGH);
      delay(2000);                      // in theory, we should not delay the loop() routine
                                        // for such a long time, but this is the last thing we do ...
      digitalWrite(pinPyro, LOW);
      timeElapsed = true;
    }
  }
}


  


